[![view on npm](https://badgen.net/npm/v/byte-size)](https://www.npmjs.org/package/byte-size)
[![npm module downloads](https://badgen.net/npm/dt/byte-size)](https://www.npmjs.org/package/byte-size)
[![Gihub repo dependents](https://badgen.net/github/dependents-repo/75lb/byte-size)](https://github.com/75lb/byte-size/network/dependents?dependent_type=REPOSITORY)
[![Gihub package dependents](https://badgen.net/github/dependents-pkg/75lb/byte-size)](https://github.com/75lb/byte-size/network/dependents?dependent_type=PACKAGE)
[![Node.js CI](https://github.com/75lb/byte-size/actions/workflows/node.js.yml/badge.svg)](https://github.com/75lb/byte-size/actions/workflows/node.js.yml)
[![js-standard-style](https://img.shields.io/badge/code%20style-standard-brightgreen.svg)](https://github.com/feross/standard)

***Upgraders, please check the [release notes](https://github.com/75lb/byte-size/releases).***

# byte-size

An isomorphic, load-anywhere function to convert a bytes value (e.g. `3456`) to a human-readable string (`'3.5 kB'`). Choose between [metric or IEC units](https://en.wikipedia.org/wiki/Gigabyte) (summarised below) or specify your own custom units.

Value  | Metric        | Metric (octet) |
-----  | ------------- | -------------- |
1000   | kB  kilobyte  | ko  kilooctet  |
1000^2 | MB  megabyte  | Mo  megaoctet  |
1000^3 | GB  gigabyte  | Go  gigaoctet  |
1000^4 | TB  terabyte  | To  teraoctet  |
1000^5 | PB  petabyte  | Po  petaoctet  |
1000^6 | EB  exabyte   | Eo  exaoctet   |
1000^7 | ZB  zettabyte | Zo  zettaoctet |
1000^8 | YB  yottabyte | Yo  yottaoctet |

Value  | IEC          | IEC (octet)   |
------ | ------------ | ------------- |
1024   | KiB kibibyte | Kio kibioctet |
1024^2 | MiB mebibyte | Mio mebioctet |
1024^3 | GiB gibibyte | Gio gibioctet |
1024^4 | TiB tebibyte | Tio tebioctet |
1024^5 | PiB pebibyte | Pio pebioctet |
1024^6 | EiB exbibyte | Eio exbioctet |
1024^7 | ZiB zebibyte | Zio zebioctet |
1024^8 | YiB yobibyte | Yio yobioctet |

## Synopsis

By default, `byteSize` converts the input number to a human readable string with metric units and a precision of 1.

```js
> const byteSize = require('byte-size')

> byteSize(1580)
{ value: '1.6', unit: 'kB', long: 'kilobytes' }
```

The object returned by `byteSize` defines a `toString` method therefore can be used directly in string context.

```js
> `Filesize: ${byteSize(12400)}`
'Filesize: 12.4 kB'
```

Override the default `toString` behaviour by setting [`options.toStringFn`](#bytesizebytes-options--object-).

```js
> function toStringFn () {
  return `**${this.value}${this.unit}**`
}

> `Filesize: ${byteSize(12400, { toStringFn })}`
'Filesize: **12.4kB**'
```

Beside the default of `metric`, there are three other built-in units available: `metric_octet`, `iec` and `iec_octet`.

```js
> byteSize(1580, { units: 'iec' })
{ value: '1.5', unit: 'KiB', long: 'kibibytes' }

> byteSize(1580, { units: 'iec_octet' })
{ value: '1.5', unit: 'Kio', long: 'kibioctets' }

> byteSize(1580, { units: 'metric_octet' })
{ value: '1.6', unit: 'ko', long: 'kilooctets' }
```

You can adjust the `precision`.

```js
> byteSize(1580, { units: 'iec', precision: 3 })
{ value: '1.543', unit: 'KiB', long: 'kibibytes' }

> byteSize(1580, { units: 'iec', precision: 0 })
{ value: '2', unit: 'KiB', long: 'kibibytes' }
```

Define custom units by passing an object containing one or more additional conversion tables to `options.customUnits`. In `options.units`, specify the name of a property from the `customUnits` object.

```js
> const customUnits = {
  simple: [
    { from: 0   , to: 1e3 , unit: ''  },
    { from: 1e3 , to: 1e6 , unit: 'K', long: 'thousand' },
    { from: 1e6 , to: 1e9 , unit: 'Mn', long: 'million' },
    { from: 1e9 , to: 1e12, unit: 'Bn', long: 'billion' }
  ]
}

> const { value, unit } = byteSize(10000, { customUnits, units: 'simple' })

> `${value}${unit}`
'10.0K'
```

Override the built-in defaults for the duration of the process by passing an options object to `byteSize.defaultOptions()`. This results in cleaner code in cases where `byteSize` is used often with the same options.

```js
> byteSize.defaultOptions({
  units: 'simple',
  precision: 2,
  customUnits: {
    simple: [
      { from: 0, to: 1e3, unit: '' },
      { from: 1e3, to: 1e6, unit: 'k' },
      { from: 1e6, to: 1e9, unit: 'm' },
      { from: 1e9, to: 1e12, unit: 'bn' },
    ]
  },
  toStringFn: function () {
    return this.value + this.unit
  }
})

> [2400, 16400, 3991200].map(byteSize).join(', ')
'2.40k, 16.40k, 3.99m'
```

<a name="module_byte-size"></a>

## byte-size

* [byte-size](#module_byte-size)
    * [byteSize(bytes, [options])](#exp_module_byte-size--byteSize) ⇒ <code>object</code> ⏏
        * [.defaultOptions(options)](#module_byte-size--byteSize.defaultOptions)

<a name="exp_module_byte-size--byteSize"></a>

### byteSize(bytes, [options]) ⇒ <code>object</code> ⏏
Returns an object with the spec `{ value: string, unit: string, long: string }`. The returned object defines a `toString` method meaning it can be used in any string context.

**Kind**: Exported function  

| Param | Type | Description |
| --- | --- | --- |
| bytes | <code>number</code> | The bytes value to convert. |
| [options] | <code>object</code> | Optional config. |
| [options.precision] | <code>number</code> | Number of decimal places. Defaults to `1`. |
| [options.units] | <code>string</code> | Specify `'metric'`, `'iec'`, `'metric_octet'`, `'iec_octet'` or the name of a property from the custom units table in `options.customUnits`. Defaults to `metric`. |
| [options.customUnits] | <code>object</code> | An object containing one or more custom unit lookup tables. |
| [options.toStringFn] | <code>function</code> | A `toString` function to override the default. |
| [options.locale] | <code>string</code> \| <code>Array.&lt;string&gt;</code> | *Node >=13 or modern browser only - on earlier platforms this option is ignored*. The locale to use for number formatting (e.g. `'de-DE'`). Defaults to your system locale. Passed directed into [Intl.NumberFormat()](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Intl/NumberFormat/NumberFormat). |

<a name="module_byte-size--byteSize.defaultOptions"></a>

#### byteSize.defaultOptions(options)
Set the default `byteSize` options for the duration of the process.

**Kind**: static method of [<code>byteSize</code>](#exp_module_byte-size--byteSize)  

| Param | Type | Description |
| --- | --- | --- |
| options | <code>object</code> | A `byteSize` options object. |


## Load anywhere

This library is compatible with Node.js, the Web and any style of module loader. It can be loaded anywhere, natively without transpilation.

Node.js (CommonJS):

```js
const byteSize = require('byte-size')
```

Node.js (ECMAScript Module):

```js
import byteSize from 'byte-size'
```

Within a modern browser ECMAScript Module:

```js
import byteSize from './node_modules/byte-size/index.js'
```

Browser global (adds `window.byteSize`):

```html
<script src="./node_modules/byte-size/dist/index.js"></script>
```

* * *

&copy; 2014-25 [Lloyd Brookes](https://github.com/75lb) \<opensource@75lb.com\>.

Tested by [test-runner](https://github.com/test-runner-js/test-runner). Documented by [jsdoc-to-markdown](https://github.com/jsdoc2md/jsdoc-to-markdown).
