import { MediaTypes } from '@swagger-api/apidom-core';
import { Namespace } from '@swagger-api/apidom-core';
import { ParseResultElement } from '@swagger-api/apidom-core';
import { Tree as Tree_2 } from 'web-tree-sitter';
import { Tree as Tree_3 } from 'tree-sitter';

/**
 * This version of syntactic analysis translates TreeSitter CTS
 * directly into ApiDOM.
 *
 * Transient transformation of TreeSitter CST is performed
 * using TreeSitter cursor. TreeSitter cursor is a stateful object
 * that allows us to walk syntax tree containing large number of nodes
 * with maximum efficiency. Using this transient CST transformation
 * gives us double the performance when syntactically analyzing
 * CST into ApiDOM.
 *
 * Single traversal pass is needed to get from CST to ApiDOM.
 * @public
 */
declare const analyze: (cst: Tree_3 | Tree_2, { sourceMap }?: {
    sourceMap?: boolean | undefined;
}) => ParseResultElement;
export { analyze as syntacticAnalysis }
export { analyze as syntacticAnalysisDirect }

/**
 * @public
 */
export declare const detect: (source: string) => Promise<boolean>;

/**
 * @public
 */
export declare const detectionRegExp: RegExp;

/**
 * @public
 */
export declare class JSONMediaTypes extends MediaTypes<string> {
    latest(): string;
}

/**
 * Lexical Analysis of source string using WebTreeSitter.
 * This is WebAssembly version of TreeSitters Lexical Analysis.
 *
 * Given JavaScript doesn't support true parallelism, this
 * code should be as lazy as possible and temporal safety should be fine.
 * @public
 */
export declare const lexicalAnalysis: (source: string) => Promise<Tree_2>;

/**
 * @public
 */
export declare const mediaTypes: JSONMediaTypes;

/**
 * @public
 */
export declare const namespace: Namespace;

/**
 * @public
 */
export declare const parse: ParseFunction;

/**
 * @public
 */
export declare type ParseFunction = (source: string, options?: ParseFunctionOptions) => Promise<ParseResultElement>;

/**
 * @public
 */
export declare interface ParseFunctionOptions {
    sourceMap?: boolean;
    syntacticAnalysis?: 'direct' | 'indirect';
}

/**
 * This version of syntactic analysis does following transformations:
 *   `TreeSitter CST -> JSON AST -> ApiDOM`
 *
 * Transient transformation of TreeSitter CST is performed
 * using TreeSitter cursor. TreeSitter cursor is a stateful object
 * that allows us to walk syntax tree containing large number of nodes
 * with maximum efficiency. Using this transient CST transformation
 * gives us double the performance when syntactically analyzing
 * CST into JSON AST.
 *
 * Two traversals passes are needed to get from CST to ApiDOM.
 * This analysis is much slower than the direct one, but allows
 * to do additional analysis magic on JSON AST.
 * @public
 */
export declare const syntacticAnalysisIndirect: (cst: Tree, { sourceMap }?: {
    sourceMap?: boolean | undefined;
}) => ParseResultElement;

/**
 * @public
 */
export declare type Tree = Tree_2 | Tree_3;

export { }
